<?php
/**
 * Fetch all MT5 trades (Buy/Sell) date-wise for all users or by filters.
 * Works with MT5 Web API connection (see connection.php + mt5_api.php).
 */

header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization");

include_once __DIR__ . "/connection.php";

// ============= READ INPUT =============
$input = json_decode(file_get_contents("php://input"), true);

$type       = $input['type']       ?? 'All'; // All | Buy | Sell
$group      = $input['group']      ?? 'All';
$symbol     = $input['symbol']     ?? 'All';
$start_date = $input['start_date'] ?? date('Y-m-d', strtotime('-30 days'));
$end_date   = $input['end_date']   ?? date('Y-m-d');

// Convert to UTC timestamps
$start_timestamp = strtotime($start_date . ' 00:00:00 UTC');
$end_timestamp   = strtotime($end_date . ' 23:59:59 UTC');

// ============= CONNECT TO MT5 =============
$mt5 = new MT5Connection();
if (!$mt5->connect()) {
    echo json_encode([
        "success" => false,
        "error"   => "MT5 connection failed",
        "details" => $mt5->lastError()
    ], JSON_PRETTY_PRINT);
    exit;
}

// ============= FETCH USER LOGINS =============
$logins = [];

$total_groups = 0;
if ($group === 'All') {
    if ($mt5->api->GroupTotal($total_groups) && $total_groups > 0) {
        for ($i = 0; $i < $total_groups; $i++) {
            $group_obj = null;
            if ($mt5->api->GroupNext($i, $group_obj) && !empty($group_obj->name)) {
                $group_logins = [];
                $mt5->api->UserLogins($group_obj->name, $group_logins);
                if (!empty($group_logins)) {
                    $logins = array_merge($logins, $group_logins);
                }
            }
        }
    }
} else {
    $mt5->api->UserLogins($group, $logins);
}

if (empty($logins)) {
    echo json_encode([
        "success" => true,
        "message" => "No user logins found.",
        "total_trades" => 0,
        "trades" => []
    ], JSON_PRETTY_PRINT);
    exit;
}

// ============= FETCH DEALS FOR EACH USER =============
$all_trades   = [];
$total_pl     = 0;
$total_volume = 0;
$total_swaps  = 0;

foreach ($logins as $login) {
    $total_deals = 0;
    if (!$mt5->api->DealGetTotal($login, $start_timestamp, $end_timestamp, $total_deals)) {
        continue; // skip on error
    }

    if ($total_deals <= 0) continue;

    $deals = [];
    if (!$mt5->api->DealGetPage($login, $start_timestamp, $end_timestamp, 0, $total_deals, $deals)) {
        continue;
    }

    foreach ($deals as $deal) {
        // Only Buy/Sell deals (action 0=Buy, 1=Sell)
        if (!in_array($deal->action, [0, 1])) continue;

        // Filter by symbol if needed
        if ($symbol !== 'All' && $deal->symbol !== $symbol) continue;

        $trade_type = ($deal->action == 0) ? 'Buy' : 'Sell';
        if ($type !== 'All' && strcasecmp($type, $trade_type) !== 0) continue;

        $trade_date = gmdate('Y-m-d', $deal->time_msc / 1000); // group by date (UTC)

        $all_trades[] = [
            "ticket"      => $deal->ticket,
            "login"       => $login,
            "symbol"      => $deal->symbol,
            "type"        => $trade_type,
            "volume"      => $deal->volume,
            "price"       => $deal->price,
            "profit"      => $deal->profit,
            "swap"        => $deal->swap,
            "commission"  => $deal->commission,
            "trade_date"  => $trade_date,
            "open_time"   => gmdate('Y-m-d H:i:s', $deal->time_msc / 1000)
        ];

        $total_pl     += $deal->profit;
        $total_volume += $deal->volume;
        $total_swaps  += $deal->swap;
    }
}

// ============= SORT TRADES BY DATE =============
usort($all_trades, function ($a, $b) {
    return strtotime($a['trade_date']) - strtotime($b['trade_date']);
});

// ============= FINAL RESPONSE =============
echo json_encode([
    "success"       => true,
    "total_trades"  => count($all_trades),
    "total_profit"  => round($total_pl, 2),
    "total_volume"  => round($total_volume, 2),
    "total_swaps"   => round($total_swaps, 2),
    "trades"        => $all_trades
], JSON_PRETTY_PRINT);
?>
