<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization");

include_once __DIR__ . "/config.php";
include_once __DIR__ . "/connection.php";
include_once __DIR__ . "/mt5_api/mt5_api.php"; // MT5 API

$input = json_decode(file_get_contents("php://input"), true);
$email = $input["email"] ?? null;

if (!$email) {
    echo json_encode(["success"=>false,"message"=>"Email is required","data"=>[]]);
    exit;
}

/* -------------------- BFS: collect all linked IB users -------------------- */
$allLinkedEmails = [];
$queue = [$email];

while (!empty($queue)) {
    $current = array_shift($queue);

    $ib_fields = [];
    for ($i = 1; $i <= 20; $i++) $ib_fields[] = "ib$i = ?";
    $ib_where = implode(" OR ", $ib_fields);

    $stmt = $conn->prepare("SELECT email FROM loginusers WHERE $ib_where");
    if (!$stmt) continue;

    $types = str_repeat("s", 20);
    $params = array_fill(0, 20, $current);
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    $result = $stmt->get_result();

    while ($row = $result->fetch_assoc()) {
        $childEmail = $row['email'];
        if (!in_array($childEmail, $allLinkedEmails, true)) {
            $allLinkedEmails[] = $childEmail;
            $queue[] = $childEmail;
        }
    }
}

/* -------------------- Fetch all live accounts for linked emails -------------------- */
$live_by_login = [];
$stmt = $conn->prepare("
    SELECT la.*, lu.first_name, lu.last_name 
    FROM liveaccounts la
    LEFT JOIN loginusers lu ON la.user_id = lu.id
    WHERE la.email = ?
");

foreach ($allLinkedEmails as $userEmail) {
    $stmt->bind_param("s", $userEmail);
    $stmt->execute();
    $result = $stmt->get_result();

    while ($row = $result->fetch_assoc()) {
        $loginKey = (string)($row['trade_id'] ?? '');
        if ($loginKey === '') {
            $loginKey = $row['email'] . '_' . $row['account_type'] . '_' . uniqid();
        }
        if (!isset($live_by_login[$loginKey])) {
            $live_by_login[$loginKey] = $row;
        }
    }
}

$total_live_accounts = count($live_by_login);

/* -------------------- Determine Rank based on active clients -------------------- */
$totalActiveClients = count($allLinkedEmails);
$rankNumber = 1;
$rankName = "Rank 1";
$rankData = [];

$stmt_rank = $conn->prepare("SELECT * FROM ranks WHERE min_active_clients <= ? ORDER BY rank_number DESC LIMIT 1");
$stmt_rank->bind_param("i", $totalActiveClients);
$stmt_rank->execute();
$res_rank = $stmt_rank->get_result();
if ($row_rank = $res_rank->fetch_assoc()) {
    $rankNumber = intval($row_rank['rank_number']);
    $rankName = $row_rank['rank_name'];
    $rankData = $row_rank;
}
$stmt_rank->close();

/* -------------------- Initialize totals -------------------- */
$mt5_data = [];
$total_volume = 0.0;
$total_rebate = 0.0;
$total_deposits = 0.0;
$total_withdrawals = 0.0;

/* -------------------- Connect MT5 -------------------- */
$mt5 = new MTWebAPI("WebAPITesterLocal", "./logs", true);
$ret = $mt5->Connect(MT5_SERVER_IP, MT5_SERVER_PORT, 5, MT5_SERVER_WEB_LOGIN, MT5_SERVER_WEB_PASSWORD);
if ($ret != MTRetCode::MT_RET_OK) {
    echo json_encode(["success" => false, "message" => "MT5 connection failed"]);
    exit;
}

/* -------------------- Process each live account -------------------- */
foreach ($live_by_login as $row) {
    $login = (string)($row["trade_id"] ?? '');
    $userEmail = $row["email"];
    $trade_id = intval($row['trade_id'] ?? 0);
    $userFullName = trim(($row['first_name'] ?? '') . ' ' . ($row['last_name'] ?? ''));
    $userFullName = $userFullName ?: ($row["account_name"] ?? '');
    $account_type_name = $row['account_type'] ?? 'Unknown';

    // --- Determine tier/level for this account ---
    $tierLevel = 0;
    for ($i = 1; $i <= 20; $i++) {
        if (($row["ib$i"] ?? '') === $email) {
            $tierLevel = $i;
            break;
        }
    }

    // --- Fetch deposit & withdraw per trade_id ---
    $stmtDeposit = $conn->prepare("SELECT SUM(deposit_amount) as totalDeposit FROM wallet_deposit WHERE email=? AND trade_id=? AND Status=1");
    $stmtDeposit->bind_param("si", $userEmail, $trade_id);
    $stmtDeposit->execute();
    $depositRow = $stmtDeposit->get_result()->fetch_assoc();
    $account_deposit = floatval($depositRow['totalDeposit'] ?? 0);
    $stmtDeposit->close();

    $stmtWithdraw = $conn->prepare("SELECT SUM(withdraw_amount) as totalWithdraw FROM wallet_withdraw WHERE email=? AND trade_id=? AND Status=1");
    $stmtWithdraw->bind_param("si", $userEmail, $trade_id);
    $stmtWithdraw->execute();
    $withdrawRow = $stmtWithdraw->get_result()->fetch_assoc();
    $account_withdraw = floatval($withdrawRow['totalWithdraw'] ?? 0);
    $stmtWithdraw->close();

    $total_deposits += $account_deposit;
    $total_withdrawals += $account_withdraw;

    // --- Fetch balance from MT5 ---
    $balance = 0;
    $account_info = null;
    if ($mt5->UserAccountGet($trade_id, $account_info) == MTRetCode::MT_RET_OK) {
        $balance = floatval($account_info->Balance ?? 0);
    }

    // --- Fetch volume from MT5 ---
    $volume = 0;
    $totalDeals = 0;
    $ret_total = $mt5->DealGetTotal($trade_id, 0, 0, $totalDeals);
    if ($ret_total == MTRetCode::MT_RET_OK && $totalDeals > 0) {
        $deals = [];
        $ret_page = $mt5->DealGetPage($trade_id, 0, 0, 0, $totalDeals, $deals);
        if ($ret_page == MTRetCode::MT_RET_OK) {
            foreach ($deals as $deal) {
                $volume += isset($deal->VolumeClosed) ? $deal->VolumeClosed * 0.0001 : 0;
            }
        }
    }

    /* -------------------- Calculate rebate like tiers code -------------------- */
    $rebate = 0;
    $rank_commission = 0;

    // Determine valid account type from account_types table
    $isValidAccount = false;
    $account_group = '';
    if (!empty($account_type_name)) {
        $stmt_type = $conn->prepare("SELECT ac_name, ac_group FROM account_types WHERE ac_name = ?");
        $stmt_type->bind_param("s", $account_type_name);
        $stmt_type->execute();
        $res_type = $stmt_type->get_result();
        if ($typeRow = $res_type->fetch_assoc()) {
            $account_type_name = $typeRow['ac_name'];
            $account_group = $typeRow['ac_group'];
            $isValidAccount = true;

            // Build commission field from ranks table
            $commission_type = strtolower($account_type_name);
            $commission_field_type = "level_{$tierLevel}_{$commission_type}_commission";

            $rank_commission = isset($rankData[$commission_field_type])
                ? floatval($rankData[$commission_field_type])
                : floatval($rankData["level_{$tierLevel}_basic_commission"] ?? 0);
        }
        $stmt_type->close();
    }

    if ($isValidAccount && $volume > 0) {
        $rebate = $volume * $rank_commission;
    }

    // --- Collect account data ---
    $mt5_data[] = [
        "login"        => $login,
        "name"         => $userFullName,
        "account_type" => $account_type_name,
        "account_group"=> $account_group,
        "email"        => $userEmail,
        "tier"         => $tierLevel,
        "balance"      => round($balance, 2),
        "deposit"      => round($account_deposit, 2),
        "withdraw"     => round($account_withdraw, 2),
        "volume"       => round($volume, 4),
        "rebate"       => round($rebate, 2),
        "rank_commission" => $rank_commission
    ];

    $total_volume += $volume;
    $total_rebate += $rebate;
}

/* -------------------- Disconnect MT5 -------------------- */
$mt5->Disconnect();

/* -------------------- Response -------------------- */
echo json_encode([
    "success" => true,
    "message" => "All linked IB accounts fetched successfully",
    "data" => [
        "email"               => $email,
        "all_linked_emails"   => array_values($allLinkedEmails),
        "total_live_accounts" => $total_live_accounts,
        "total_deposits"      => round($total_deposits, 2),
        "total_withdrawals"   => round($total_withdrawals, 2),
        "total_volume"        => round($total_volume, 4),
        "total_rebate"        => round($total_rebate, 2),
        "rank"                => [
            "rank_number" => $rankNumber,
            "rank_name" => $rankName,
            "active_clients" => $totalActiveClients
        ],
        "mt5_accounts"        => array_values($mt5_data)
    ]
], JSON_PRETTY_PRINT);

exit;
?>
