<?php
// Simplified email test script that only tests the mail system without database dependencies

// Include test configuration
require_once 'test_mail_config.php';

// Include only the mail system
require_once 'mail_system/integrated_mail_system.php';

// Test email address
$TEST_EMAIL = 'sagarnandal95@gmail.com';
$TEST_NAME = 'Sagar Nandal';

echo "📧 ZForex Mail System - All Templates Test\n";
echo "==========================================\n";
echo "Testing email: $TEST_EMAIL\n";
echo "Testing name: $TEST_NAME\n";
echo "==========================================\n\n";

// Get mail system instance
$mailSystem = getMailSystem();

// Test results array
$testResults = [];
$testCount = 0;
$successCount = 0;

// Test all 22 templates
$tests = [
    // 1. Register Verification
    'register_verification' => function() use ($mailSystem, $TEST_EMAIL, $TEST_NAME) {
        return $mailSystem->sendRegisterVerification($TEST_EMAIL, $TEST_NAME, 'VER123456', '24 hours');
    },
    
    // 2. After Verify Credentials
    'after_verify_credentials' => function() use ($mailSystem, $TEST_EMAIL, $TEST_NAME) {
        return $mailSystem->sendAfterVerifyCredentials($TEST_EMAIL, $TEST_NAME, 'USR789012');
    },
    
    // 3. Login Alert
    'login_alert' => function() use ($mailSystem, $TEST_EMAIL, $TEST_NAME) {
        return $mailSystem->sendLoginAlert($TEST_EMAIL, $TEST_NAME, '192.168.1.100', 'Chrome on Windows');
    },
    
    // 4. Live Account Opened
    'open_live_account' => function() use ($mailSystem, $TEST_EMAIL, $TEST_NAME) {
        return $mailSystem->sendLiveAccountOpened($TEST_EMAIL, $TEST_NAME, 'ACC123456789', 'Standard', '100', 'USD');
    },
    
    // 5. KYC Upload
    'kyc_upload' => function() use ($mailSystem, $TEST_EMAIL, $TEST_NAME) {
        return $mailSystem->sendKYCUpload($TEST_EMAIL, $TEST_NAME, 'KYC987654', 'ID Document, Address Proof');
    },
    
    // 6. KYC Approved
    'kyc_approved' => function() use ($mailSystem, $TEST_EMAIL, $TEST_NAME) {
        return $mailSystem->sendKYCApproved($TEST_EMAIL, $TEST_NAME, 'Full', 'September 30, 2025');
    },
    
    // 7. Deposit Request Submitted
    'deposit_request_submitted' => function() use ($mailSystem, $TEST_EMAIL, $TEST_NAME) {
        return $mailSystem->sendDepositRequestSubmitted($TEST_EMAIL, $TEST_NAME, 1000, 'USD', 'Bank Transfer', 'DEP123456');
    },
    
    // 8. Deposit Approved
    'deposit_approved' => function() use ($mailSystem, $TEST_EMAIL, $TEST_NAME) {
        return $mailSystem->sendDepositApproved($TEST_EMAIL, $TEST_NAME, 1000, 'USD', 5000, 'TXN789012');
    },
    
    // 9. Deposit Failed
    'deposit_failed' => function() use ($mailSystem, $TEST_EMAIL, $TEST_NAME) {
        return $mailSystem->sendDepositFailed($TEST_EMAIL, $TEST_NAME, 500, 'USD', 'Insufficient funds', 'DEP345678');
    },
    
    // 10. Withdrawal OTP
    'otp_withdrawal' => function() use ($mailSystem, $TEST_EMAIL, $TEST_NAME) {
        return $mailSystem->sendWithdrawalOTP($TEST_EMAIL, $TEST_NAME, '987654', 500, 'USD', 'Bank Account ****1234', '10 minutes');
    },
    
    // 11. Withdrawal Request Submitted
    'withdrawal_request_submitted' => function() use ($mailSystem, $TEST_EMAIL, $TEST_NAME) {
        return $mailSystem->sendWithdrawalRequestSubmitted($TEST_EMAIL, $TEST_NAME, 500, 'USD', 'Bank Transfer', 'WTH123456');
    },
    
    // 12. Withdrawal Approved
    'withdrawal_approved' => function() use ($mailSystem, $TEST_EMAIL, $TEST_NAME) {
        return $mailSystem->sendWithdrawalApproved($TEST_EMAIL, $TEST_NAME, 500, 'USD', 'TXN987654', '2-3 business days', 4500);
    },
    
    // 13. Withdrawal Rejected
    'withdrawal_rejected' => function() use ($mailSystem, $TEST_EMAIL, $TEST_NAME) {
        return $mailSystem->sendWithdrawalRejected($TEST_EMAIL, $TEST_NAME, 500, 'USD', 'Invalid bank details', 'WTH789012');
    },
    
    // 14. IB Commission Withdrawal
    'ib_commission_withdrawal' => function() use ($mailSystem, $TEST_EMAIL, $TEST_NAME) {
        return $mailSystem->sendIBCommissionWithdrawal($TEST_EMAIL, $TEST_NAME, 150, 'USD', 'September 2025', 5);
    },
    
    // 15. Referral Registration
    'referral_registration' => function() use ($mailSystem, $TEST_EMAIL, $TEST_NAME) {
        return $mailSystem->sendReferralRegistration($TEST_EMAIL, $TEST_NAME, 'John Smith', '5');
    },
    
    // 16. Trade Opened
    'trade_opened' => function() use ($mailSystem, $TEST_EMAIL, $TEST_NAME) {
        return $mailSystem->sendTradeOpened($TEST_EMAIL, $TEST_NAME, 'EUR/USD', 'BUY', '0.1', '1.0850');
    },
    
    // 17. Trade Closed
    'trade_closed' => function() use ($mailSystem, $TEST_EMAIL, $TEST_NAME) {
        return $mailSystem->sendTradeClosed($TEST_EMAIL, $TEST_NAME, 'EUR/USD', '0.1', '1.0800', '1.0900', 10.50, 'USD', '2 hours');
    },
    
    // 18. Daily Trade Summary
    'daily_trade_summary' => function() use ($mailSystem, $TEST_EMAIL, $TEST_NAME) {
        return $mailSystem->sendDailyTradeSummary($TEST_EMAIL, $TEST_NAME, 'September 30, 2025', 15, 10, 5, 125.75, 'USD', 5125.75);
    },
    
    // 19. Password Reset
    'password_reset' => function() use ($mailSystem, $TEST_EMAIL, $TEST_NAME) {
        return $mailSystem->sendPasswordReset($TEST_EMAIL, $TEST_NAME, 'RESET123456', '1 hour');
    },
    
    // 20. Password Changed
    'password_changed' => function() use ($mailSystem, $TEST_EMAIL, $TEST_NAME) {
        return $mailSystem->sendPasswordChanged($TEST_EMAIL, $TEST_NAME, '192.168.1.100');
    },
    
    // 21. Profile Updated
    'profile_updated' => function() use ($mailSystem, $TEST_EMAIL, $TEST_NAME) {
        return $mailSystem->sendProfileUpdated($TEST_EMAIL, $TEST_NAME, 'Email, Phone Number');
    },
    
    // 22. Monthly Statement
    'monthly_statement' => function() use ($mailSystem, $TEST_EMAIL, $TEST_NAME) {
        return $mailSystem->sendMonthlyStatement($TEST_EMAIL, $TEST_NAME, 'September', '2025', 125, 567.89, 'USD', 10000, 5000, 15678.89);
    }
];

echo "🧪 Running Tests for All 22 Email Templates\n";
echo "============================================\n";

// Run all tests
foreach ($tests as $templateName => $testFunction) {
    $testCount++;
    echo "Test $testCount: $templateName\n";
    
    try {
        $result = $testFunction();
        $testResults[$templateName] = $result;
        
        if ($result['success']) {
            $successCount++;
            echo "  ✅ PASSED: " . $result['message'] . "\n";
        } else {
            echo "  ❌ FAILED: " . $result['message'] . "\n";
        }
    } catch (Exception $e) {
        $testResults[$templateName] = ['success' => false, 'message' => $e->getMessage()];
        echo "  ❌ EXCEPTION: " . $e->getMessage() . "\n";
    }
    
    // Add a small delay between emails
    sleep(1);
}

// Summary
echo "\n📊 Test Summary\n";
echo "===============\n";

foreach ($testResults as $templateName => $result) {
    $status = $result['success'] ? 'PASSED' : 'FAILED';
    echo sprintf("%-30s %s\n", $templateName, $status);
}

$successRate = ($testCount > 0) ? round(($successCount / $testCount) * 100, 2) : 0;

echo "\nTotal Tests: $testCount\n";
echo "Successful: $successCount\n";
echo "Failed: " . ($testCount - $successCount) . "\n";
echo "Success Rate: {$successRate}%\n";

echo "\n📧 Next Steps\n";
echo "=============\n";
echo "Please check your email inbox at $TEST_EMAIL to verify that all emails were delivered successfully.\n";
echo "If any emails failed to send, check the error messages above and verify your mail configuration in mail_system/config.php.\n";
?>